/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.cie.gdr.nio;

import java.io.IOException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

public class NioFileLockService {
    private static final Logger _log = Logger.getLogger(NioFileLockService.class.getName());
    private static Map<Path, FileLockHolder> _fileLockHolders = new HashMap<Path, FileLockHolder>();

    public static boolean getFileLock(Path path) throws IOException {
        return NioFileLockService.getLock(path, true, false);
    }

    public static boolean getFileLockWithLimitedWait(Path path, int attempts, int waitTime) throws IOException {
        return NioFileLockService.getLockWithLimitedWait(path, attempts, waitTime, false);
    }

    private static boolean getLockWithLimitedWait(Path path, int attempts, int waitTime, boolean shared) throws IOException {
        try {
            int n = attempts = attempts <= 0 ? 1 : attempts;
            while (attempts-- > 0) {
                boolean result = NioFileLockService.getLock(path, false, shared);
                if (result) {
                    return true;
                }
                try {
                    Thread.sleep(waitTime);
                }
                catch (InterruptedException ie) {
                    _log.info("Thread was interrupted. Will start processing again");
                    Thread.currentThread().interrupt();
                }
            }
        }
        catch (OverlappingFileLockException e) {
            _log.log(Level.WARNING, e.getMessage(), e);
            return false;
        }
        return false;
    }

    public static boolean getFileLock(Path path, boolean wait) throws IOException {
        return NioFileLockService.getLock(path, wait, false);
    }

    private static boolean getLock(Path path, boolean wait, boolean shared) throws IOException {
        _log.finer("Attempting to lock file: " + path + " wait: " + wait + " shared: " + shared);
        boolean result = NioFileLockService.getFileLockHolder(path, true).lockFile(wait, shared);
        _log.finer("File locking result for file:" + path + " result: " + result);
        return result;
    }

    public static boolean getSharedLock(Path path) throws IOException {
        return NioFileLockService.getLock(path, true, true);
    }

    public static boolean getSharedLock(Path path, boolean wait) throws IOException {
        return NioFileLockService.getLock(path, wait, true);
    }

    public static boolean getSharedLockWithLimitedWait(Path path, int attempts, int waitTime) throws IOException {
        return NioFileLockService.getLockWithLimitedWait(path, attempts, waitTime, true);
    }

    public static void releaseFileLock(Path path) throws IOException {
        _log.finer("Attempting to release file lock for path: " + path);
        FileLockHolder lockHolder = NioFileLockService.getFileLockHolder(path, false);
        if (lockHolder != null) {
            lockHolder.releaseFile();
            _log.finer("Released file lock for path: " + path);
        } else {
            _log.finer("File lock not being kept for path: " + path + ", nothing was done.");
        }
    }

    public static void resetFileLocks() throws IOException {
        _log.finer("Attempting to release all file locks");
        for (Path path : _fileLockHolders.keySet()) {
            NioFileLockService.releaseFileLock(path);
        }
    }

    public static FileChannel getFileChannel(Path path) throws IOException {
        return NioFileLockService.getFileLockHolder(path, true).getFileChannel();
    }

    public static void closeFileChannel(Path path) throws IOException {
        FileLockHolder flh = NioFileLockService.getFileLockHolder(path, false);
        if (flh != null) {
            flh.closeFileChannel();
        }
    }

    protected static synchronized FileLockHolder getFileLockHolder(Path path, boolean create) {
        FileLockHolder lockHolder = _fileLockHolders.get(path);
        if (create && lockHolder == null) {
            lockHolder = new FileLockHolder(path);
            _fileLockHolders.put(path, lockHolder);
        }
        return lockHolder;
    }

    static class FileLockHolder {
        private Path _path;
        private FileChannel _fileChannel;
        private FileLock _lock;

        FileLockHolder(Path path) {
            this._path = path;
        }

        public Path getPath() {
            return this._path;
        }

        public boolean lockFile(boolean wait) throws IOException {
            return this.lockFile(wait, false);
        }

        public synchronized boolean lockFile(boolean wait, boolean shared) throws IOException {
            if (this._lock == null || !this._lock.isValid()) {
                if (_log.isLoggable(Level.FINER)) {
                    _log.finer("Locking path: " + this.getPath() + (shared ? " shared" : " exclusive"));
                }
                FileChannel fileChannel = this.getFileChannel();
                this._lock = wait ? fileChannel.lock(0L, Long.MAX_VALUE, shared) : fileChannel.tryLock(0L, Long.MAX_VALUE, shared);
                _log.finer("lock attempt returned:" + this._lock);
                return this._lock != null;
            }
            if (this._lock.isShared()) {
                _log.finer("This JVM already holds a shared lock on file " + this.getPath());
                if (shared) {
                    return true;
                }
                _log.warning("Can't get an exclusive lock because this JVM already holds a shared lock");
                return false;
            }
            _log.finer("This JVM already holds an exclusive lock on file " + this.getPath());
            return true;
        }

        public synchronized void releaseFile() throws IOException {
            if (this._lock != null) {
                if (_log.isLoggable(Level.INFO)) {
                    _log.finer("Releasing lock for file: " + this.getPath());
                }
                this._lock.release();
                this._lock = null;
            }
            this.closeFileChannel();
        }

        public FileChannel getFileChannel() throws IOException {
            if (this._fileChannel == null) {
                Path path = this.getPath();
                Files.createDirectories(path.getParent(), new FileAttribute[0]);
                if (_log.isLoggable(Level.INFO)) {
                    _log.finer("Creating random access file for: " + path);
                }
                this._fileChannel = FileChannel.open(path, StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
            }
            return this._fileChannel;
        }

        public void closeFileChannel() throws IOException {
            if (this._fileChannel != null) {
                if (_log.isLoggable(Level.INFO)) {
                    _log.finer("Closing file channel for: " + this.getPath());
                }
                this._fileChannel.close();
                this._fileChannel = null;
            }
        }
    }
}

